#!/bin/bash
# AWS EC2 LXC Deployment for HDGL Monolith

set -e

# Variables
INSTANCE_TYPE="t3.medium"
AMI_ID="ami-0e872aee57663ae2d"
KEY_PAIR="hdgl-key-pair"
SECURITY_GROUP="hdgl-sg"
SUBNET_ID="subnet-0a1b2c3d4e5f67890"
VPC_ID="vpc-0a1b2c3d4e5f67890"
MONOLITH_FILE="hdgl_monolith.c"
MONOLITH_CHECKSUM="sha256sum hdgl_monolith.c | cut -d' ' -f1"
EXPECTED_CHECKSUM="e4b7a1c2d3e4f5a6b7c8d9e0f1a2b3c4d5e6f7a8b9c0d1e2f3a4b5c6d7e8f9"

# Create Security Group
aws ec2 create-security-group --group-name $SECURITY_GROUP --description "HDGL LXC SG" --vpc-id $VPC_ID
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 22 --cidr 0.0.0.0/0
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 8080 --cidr 0.0.0.0/0
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 9090 --cidr 0.0.0.0/0

# Securely copy monolith file
aws s3 cp $MONOLITH_FILE s3://hdgl-bucket/$MONOLITH_FILE

# Launch Aggregator Instance
INSTANCE_ID=$(aws ec2 run-instances \
    --image-id $AMI_ID \
    --count 1 \
    --instance-type $INSTANCE_TYPE \
    --key-name $KEY_PAIR \
    --security-group-ids $SECURITY_GROUP \
    --subnet-id $SUBNET_ID \
    --user-data "#!/bin/bash
    apt-get update && apt-get install -y lxc lxc-templates build-essential libssl-dev libgmp-dev cmake libsodium-dev libsecp256k1-dev
    aws s3 cp s3://hdgl-bucket/$MONOLITH_FILE /var/lib/lxc/hdgl-aggregator/rootfs/root/
    CHECKSUM=$($MONOLITH_CHECKSUM /var/lib/lxc/hdgl-aggregator/rootfs/root/hdgl_monolith.c)
    if [ \"$CHECKSUM\" != \"$EXPECTED_CHECKSUM\" ]; then
        echo \"Checksum validation failed\" >&2
        exit 1
    fi
    lxc-create -t ubuntu -n hdgl-aggregator -- -d ubuntu -r 24.04 -a amd64
    lxc-execute -n hdgl-aggregator -- gcc -o /root/hdgl_monolith /root/hdgl_monolith.c -lssl -lcrypto -lgmp -lm -lsodium -lsecp256k1
    lxc-start -n hdgl-aggregator -d -- /root/hdgl_monolith --mode=aggregator --port=8080 --max-nodes=10000" \
    --query 'Instances[0].InstanceId' --output text)

echo "Aggregator Instance ID: $INSTANCE_ID"
aws ec2 wait instance-running --instance-ids $INSTANCE_ID

# Launch Node Instances (Auto Scaling)
aws autoscaling create-launch-template \
    --launch-template-name hdgl-node-template \
    --launch-template-data '{
        "ImageId": "'$AMI_ID'",
        "InstanceType": "t3.micro",
        "UserData": "#!/bin/bash\napt-get update && apt-get install -y lxc lxc-templates build-essential libssl-dev libgmp-dev libsodium-dev libsecp256k1-dev\naws s3 cp s3://hdgl-bucket/$MONOLITH_FILE /var/lib/lxc/hdgl-node/rootfs/root/\nCHECKSUM=$($MONOLITH_CHECKSUM /var/lib/lxc/hdgl-node/rootfs/root/hdgl_monolith.c)\nif [ \"$CHECKSUM\" != \"$EXPECTED_CHECKSUM\" ]; then\n    echo \"Checksum validation failed\" >&2\n    exit 1\nfi\nlxc-create -t ubuntu -n hdgl-node -- -d ubuntu -r 24.04 -a amd64\nlxc-execute -n hdgl-node -- gcc -o /root/hdgl_monolith /root/hdgl_monolith.c -lssl -lcrypto -lgmp -lm -lsodium -lsecp256k1\nlxc-start -n hdgl-node -d -- /root/hdgl_monolith --mode=node --aggregator-endpoint=aggregator-ip:8080"
    }'

aws autoscaling create-auto-scaling-group \
    --auto-scaling-group-name hdgl-nodes \
    --launch-template LaunchTemplateName=hdgl-node-template,Version=1 \
    --min-size 1 \
    --max-size 10000 \
    --vpc-zone-identifier $SUBNET_ID

echo "Deployment complete. Aggregator: $INSTANCE_ID, Nodes: hdgl-nodes ASG"